// mac-dialog.c

#include <Carbon.h>
#include <stdio.h>

#include "dialog.h"

/* The function pointed to by DialogEventHandler will be called for every
   dialog event received by the program's main event loop. Note that after
   calling StartDialog, it is still the application's responsibility to set
   DialogEventHandler to something reasonable. Failure to do so could be
   very ugly. */
   
void (*DialogEventHandler)( short item ) = nil;

/* A typical dialog filter. Strange that it is called a ModalFilter even
   though all my dialogs are modeless. */

ModalFilterUPP DialogFilter = nil;

/* The dialog which is currently being displayed. Note that this implementation
   does not allow one modeless dialog overlapping another--nor would it be a
   good practice to do such a thing anyway (talk about ugly GUIs). Trying to
   call StartDialog when a dialog is already up will make a relatively 
   ungraceful attempt to close the current one and open the new one seamlessly. */

DialogPtr frontDialog = nil;

/* If this is false, then the DialogFilter is just the standard Macintosh
   modal filter (so don't try to dispose it!) */
 
Boolean usingCustomFilter = false;

/* Remembers the location of dialogs on screen. */

Point savedDialogLoc[kSaveCount];

/* Remembers which dialog ID maps to each item in savedDialogLoc. */ 

int   savedDialogIndex[kSaveCount], savedDialogCount = 0;


/* Open up a modeless dialog. Takes a dialog resource ID and a ModalFilterProc
   function (or nil for the default filter) as arguments. */

void StartDialog( short which, pascal Boolean (*filter)( DialogPtr theDialog, EventRecord *theEvent, short *itemHit ) )
{
	int index;
		
	// can't do two at once //
	
	if( frontDialog != nil )
	{
		ForceCloseDialog( nil );
	}
	
	// build dialog //
	
	frontDialog = GetNewDialog( which, nil, (WindowPtr)-1 );
	
	// restore positioning
	
	for( index=0; index<savedDialogCount; index++ )
	{
		if( which == savedDialogIndex[index] )
		{
			MoveWindow( GetDialogWindow(frontDialog), savedDialogLoc[index].h, savedDialogLoc[index].v, false );
			break;
		}
	}
		
	// set up dialog filter //
	
	if( filter )
	{
		DialogFilter = NewModalFilterUPP( filter );
		usingCustomFilter = true;
	}
	else
	{
		GetStdFilterProc( &DialogFilter );
		usingCustomFilter = false;	
	}
}

/* This routine forces a dialog to close. It makes use of the assumption
   that item 1 in a dialog should always be something which causes the 
   dialog to close (OK, Cancel, Stop, Done, etc).  If this assumption 
   proves to be invalid, beep and close the dialog manually. Slightly
   dangerous if item 1 is not a closer, since it could leave things in
   an unstable state. */
   
void ForceCloseDialog( short which )
{
	if( frontDialog != nil )
	{
		DialogEventHandler( 1 ); // item 1 should always be OK, Cancel, Stop, etc...
		
		if( frontDialog != nil ) // but if not... 
		{
			SysBeep(1);          // complain
			
			FinishDialog( which ); // kill it
		}
	}
}

/* Cleans up and disposes of the current modeless dialog. Takes the dialog's
   resource ID as an argument. Pass nil if it's unknown, or you don't care
   to remember the dialog's current positioning (that's all the ID is used
   for at this point). */
   
void FinishDialog( short which )
{
	Point loc = {0, 0};
	Boolean assigned = false;
	int index;
	
	// destroy dialog 
	
	if( frontDialog != nil )
	{
		// save positioning
		
		SetPortDialogPort( frontDialog );
		LocalToGlobal( &loc );
		
		if( which != nil )
		{
			for( index=0; index<savedDialogCount; index++ )
			{
				if( which == savedDialogIndex[index] )
				{
					savedDialogLoc[index] = loc;
					assigned = true;
					break;
				}
			}

			if( !assigned && savedDialogCount <= kSaveCount )
			{
				savedDialogIndex[ savedDialogCount   ] = which;
				savedDialogLoc  [ savedDialogCount++ ] = loc;
			}
		}
		
		// kill it
		
		DialogEventHandler = nil;
		DisposeDialog( frontDialog );
		frontDialog = nil;
		
		if( usingCustomFilter )
		{
			DisposeModalFilterUPP( DialogFilter );
		}
		
		DialogFilter = nil;
	}
}

/* Blink a button. */

void FlashItem( DialogPtr theDialog, short which )
{
	short itemType;
	Rect itemRect;
	ControlHandle itemHandle;
	unsigned long worthless;
	
	GetDialogItem( theDialog, which, &itemType, (Handle*) &itemHandle, &itemRect );
	HiliteControl( itemHandle, 1 );
	QDFlushPortBuffer( GetDialogPort( theDialog ), nil );
	Delay( 8, &worthless );
	HiliteControl( itemHandle, 0 );
	QDFlushPortBuffer( GetDialogPort( theDialog ), nil );
}

void SetDialogHilite( DialogPtr pDialog, short item, short hilite )
{
	// get the item information
	short itemType;
	Rect itemRect;
	Handle hItem;
	GetDialogItem(pDialog, item, &itemType, &hItem, &itemRect);
	
	// hilite the control as requested
	HiliteControl((ControlHandle)hItem, hilite);

	// make sure the enable state on the dialog item matches
	if (hilite == 255 && !(itemType & 0x80))
		SetDialogItem(pDialog, item, itemType | 0x80, hItem, &itemRect);
	else if (hilite != 255 && (itemType & 0x80))
		SetDialogItem(pDialog, item, itemType & 0x7f, hItem, &itemRect);
}

void SetDialogValue(DialogPtr pDialog, short item, short value )
{
	// get the item information
	short itemType;
	Rect itemRect;
	Handle hItem;
	GetDialogItem(pDialog, item, &itemType, &hItem, &itemRect);
	
	// hilite the control as requested
	SetGWorld((GWorldPtr)pDialog, nil);
	SetControlValue((ControlHandle)hItem, value);
}

/* Various functions to make dialog handling a little less painful. */
/*
void SetDialogValue( DialogPtr theDialog, short what, short state )
{
	ControlHandle control;

	GetDialogItemAsControl( theDialog, what, &control );
	SetControlValue( control, state );
}

void SetDialogHilite( DialogPtr theDialog, short what, Boolean enable )
{
	ControlHandle control;

	GetDialogItemAsControl( theDialog, what, &control );
	if( enable ) ActivateControl( control );
	else         DeactivateControl( control );
}

void GetDialogText( DialogPtr whichDialog, short which, Str255 what )
{
	ControlHandle control;
	
	GetDialogItemAsControl( whichDialog, which, &control );
	GetStaticTextText( control, what );
}

void SetDialogText( DialogPtr whichDialog, short which, Str255 what )
{
	ControlHandle textControl;
	
	GetDialogItemAsControl( whichDialog, which, &textControl );
	SetStaticTextText( textControl, what, true );	
}

short GetDialogValue( DialogPtr theDialog, short what )
{
	ControlHandle control;

	GetDialogItemAsControl( theDialog, what, &control );
	return GetControlValue( control );
}
void GetDialogRect( DialogPtr theDialog, short what, Rect *itemRect )
{
	ControlHandle control;

	GetDialogItemAsControl( theDialog, what, &control );
	*itemRect = (**control).contrlRect;
}
*/
